/*
 * Copyright (C) Jan 2006 Mellanox Technologies Ltd. All rights reserved.
 *
 * This software is available to you under a choice of one of two
 * licenses.  You may choose to be licensed under the terms of the GNU
 * General Public License (GPL) Version 2, available from the file
 * COPYING in the main directory of this source tree, or the
 * OpenIB.org BSD license below:
 *
 *     Redistribution and use in source and binary forms, with or
 *     without modification, are permitted provided that the following
 *     conditions are met:
 *
 *      - Redistributions of source code must retain the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer.
 *
 *      - Redistributions in binary form must reproduce the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer in the documentation and/or other materials
 *        provided with the distribution.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 *  ConfFile.cpp - Configuration-file parsing class
 *
 *  Version: $Id: ConfFile.cpp 2752 2006-01-19 14:40:17Z mst $
 *
 */
#include <stdio.h>

#include "ConfFile.h"
#include "ParamList.h"
#include "Param.h"
#include "utils.h"

namespace std {}; using namespace std;

////////////////////////////////////////////////////////////////////////
ConfFile::ConfFile() : _plist(0)
{
} // ConfFile::ConfFile


////////////////////////////////////////////////////////////////////////
ConfFile::~ConfFile()
{
} // ConfFile::~ConfFile


////////////////////////////////////////////////////////////////////////
bool ConfFile::read(const char *fname, ParamList *plist)
{
    _plist = plist;
    bool ret;

    if      (! strcasecmp(fname + strlen(fname) - 4 , ".ini")) 
	ret = parse(fname);
    else if (! strcasecmp(fname + strlen(fname) - 4 , ".brd")) { 
	ret = _plist->readBrdFile(fname);
	if (!ret)  
	    errmsg(_plist->err());
    }
    else
	ret = errmsg("Unsupported suffix for conf file %s. Allowed: .ini , .brd\n", fname);

    return ret;
} // ConfFile::read


////////////////////////////////////////////////////////////////////////
bool ConfFile::group_start(const char *name)
{
    vector<string> gr = splitv(name, ",", true);

    switch (gr.size())
    {
    case 1:
        _cur_group = name;
        _cur_geo = Param::GEO_DEF;
        break;
    case 2:
    {
        char *endp;

        _cur_group = gr[0];
        _cur_geo = strtol(gr[1].c_str(), &endp, 0);
        if (*endp)
            return errmsg("File: %s, Line: %d - invalid GEO id (%s) for "
                          "group \"%s\"",
                          fname(), line(), gr[1].c_str(), gr[0].c_str());
        break;
    }
    default:
        return errmsg("File: %s, Line: %d - \"%s\" is invalid syntax for "
                      "group name.", fname(), line(), name);
    }
    return true;
} // ConfFile::group_start


////////////////////////////////////////////////////////////////////////
bool ConfFile::param_start(const char *name, const char *value)
{

    // printf("-D- Setting param: %s=%s;\n", name, value);

    if (_cur_group == CONF_DATA_GROUP_NAME) {
        // HACK - Currently Conf file data (such as description, 
        //  file rev, fw rev, etc ...) is NOT supported - there are no real parameters in this group, 
        //  and its format  is not set. For forward compatibility, ignore parameters in this group.
        return true;
    }

    if (_cur_group.empty()) {
        return errmsg("File: %s, Line: %d - Invalid syntax. Parameter \"%s\" defined outside a group.",
                      fname(), line(), name);
    }


    string         pname = _cur_group + ".";
    vector<string> pr = splitv(name, ",", true);
    u_int32_t      geo = _cur_geo;

    switch (pr.size())
    {
    case 1:
        pname += name;
        break;
    case 2:
    {
        char *endp;

        pname += pr[0];
        geo = strtol(pr[1].c_str(), &endp, 0);
        if (*endp)
            return errmsg("File: %s, Line: %d - invalid GEO id (%s) for "
                          "parameter \"%s\"",
                          fname(), line(), pr[1].c_str(), pr[0].c_str());
        break;
    }
    default:
        return errmsg("File: %s, Line: %d - \"%s\" is invalid syntax for "
                      "parameter name.", fname(), line(), name);
    }

    map<string, Param*>::iterator pit = _plist->params.find(pname);
    if (pit == _plist->params.end())
        return errmsg("File: %s, Line: %d - parameter \"%s\" not found.",
                      fname(), line(), pname.c_str());

    Param  *par = pit->second;

    if (geo != Param::GEO_DEF  &&  !par->allow_geo)
        return errmsg("File: %s, Line: %d - parameter \"%s\" has non default\n"
                      "GEO adress (0x%x), althought it doesn't accept it.",
                      fname(), line(), pname.c_str(), geo);

    string v(value);
    v = v.substr(0, v.find_last_not_of(" \n\r\t")+1);  // truncate spaces

    if (!par->assign(v, string("file \"") + fname() + "\"", line(), geo, true))
        return errmsg("File: %s, Line: %d - Parameter assignment (%s = %s) failed - %s", 
                      fname(), 
                      line(), 
                      pname.c_str(),
                      v.c_str(),
                      par->err());

    return true;
} // ConfFile::param_start
